#include "sdk705.h"
#include "simple-wave-writer.h"

#include <cstdlib>
#include <iostream>
#include <atomic>
#include <chrono>
#include <thread>
#include <cstring>

// -----------------------------------------------------------------------------------------------------

std::atomic<bool> app_stop_flag{false};

// -----------------------------------------------------------------------------------------------------
//   Ctrl+C / Ctrl+Break / WM_CLOSE  handler
// -----------------------------------------------------------------------------------------------------

#if defined( WIN32 )||defined( _WIN32 )||defined( __WIN32__ )
#include  <Windows.h>

BOOL  __stdcall  SignalHandlerEntry( DWORD sigflag )
{
	switch( sigflag ){
		case CTRL_C_EVENT       : // SIGINT
		case CTRL_CLOSE_EVENT   : // SIGQUIT
		case CTRL_BREAK_EVENT   : // SIGTERM
		case CTRL_LOGOFF_EVENT  : // SIGLOGOFF
		case CTRL_SHUTDOWN_EVENT: // SIGSHTDWN
			app_stop_flag.store(true);
			return  TRUE;
	}

	return FALSE;   // do not disturb regular Win32 signal handling
}
#elif defined( __linux )||defined( __linux__ )
#include <csignal>

void SignalHandlerEntry(int){
	app_stop_flag.store(true);
}
#endif


int main(int argc, char *argv[])
{
#if defined( WIN32 )||defined( _WIN32 )||defined( __WIN32__ )
	// назначим хандлер сигналов для выхода из приложения
	::SetConsoleCtrlHandler( SignalHandlerEntry, true );
#elif defined( __linux )||defined( __linux__ )
	struct sigaction sigIntHandler{};
	sigIntHandler.sa_handler = SignalHandlerEntry;
	sigemptyset( &sigIntHandler.sa_mask );
	sigIntHandler.sa_flags = 0;
	sigaction( SIGINT, &sigIntHandler, nullptr );
#endif

	std::string host_addr = "192.168.1.1";
	uint16_t host_port = 15001;
	std::string broadcast_addr = "192.168.1.255";
	uint16_t broadcast_port = 15001;

	bool write_wave_file = false;
	std::string file_name;
	uint32_t channel_testing = 0;
	bool get_only_dev_list = false;

	for (int i = 1; i < argc; ++i)
	{
		if (strcmp(argv[i], "-h") == 0 || strcmp(argv[i], "--help") == 0)
		{
			std::cout <<
			          "Usage: admin_main [OPTIONS]...\n"
			          "Allowed options:\n"
			          "    -h, --help\n"
			          "        print this message and exit\n"
			          "    --host-addr\n"
			          "        host ip address\n"
			          "    --host-port\n"
			          "        host port\n"
			          "    --broadcast-addr\n"
			          "        broadcast ip address\n"
			          "    --broadcast-port\n"
			          "        broadcast port\n"
			          "    --write-file\n"
			          "        write audio log from TR-705 RTP stream\n"
			          "    --file-name\n"
			          "        audio log file name\n"
			          "    --channel-testing\n"
			          "        choose which channel we are writing in audio file\n"
			          "    --dev-list\n"
			          "        choose which channel we are testing\n";
			return EXIT_SUCCESS;
		} else if (strcmp(argv[i], "--host-addr") == 0) {
			if (argc > i + 1)
				host_addr = argv[i + 1];
		} else if (strcmp(argv[i], "--host-port") == 0) {
			if (argc > i + 1)
				host_port = uint16_t(atoi(argv[i + 1]));
		} else if (strcmp(argv[i], "--broadcast-addr") == 0) {
			if (argc > i + 1)
				broadcast_addr = argv[i + 1];
		} else if (strcmp(argv[i], "--broadcast-port") == 0) {
			if (argc > i + 1)
				broadcast_port = uint16_t(atoi(argv[i + 1]));
		} else if (strcmp(argv[i], "--write-file") == 0) {
			write_wave_file = true;
		} else if (strcmp(argv[i], "--file-name") == 0) {
			if (argc > i + 1)
				file_name = argv[i + 1];
		} else if (strcmp(argv[i], "--channel-testing") == 0) {
			if (argc > i + 1)
				channel_testing = uint32_t(atoi(argv[i + 1]));
		} else if (strcmp(argv[i], "--dev-list") == 0) {
			get_only_dev_list = true;
		}
	}

	// Получение списка устройств
	std::cout << "Getting a list of devices..." << std::endl;
	FoxxWire::SDK::DevicesList device_list;
	if (!FindActiveDevices(host_addr, host_port, broadcast_addr, broadcast_port, device_list))
		return EXIT_FAILURE;
	if (device_list.empty()) {
		std::cout << "Devices list is empty" <<std::endl;
		return EXIT_SUCCESS;
	}

	std::cout << "Devices list:" <<std::endl;
	for(const auto& dev_info : device_list)
		std::cout << dev_info.UnitId << " " << dev_info.UnitAddr << " " << dev_info.UnitPort << std::endl << std::endl;

	if(get_only_dev_list)
		return EXIT_SUCCESS;

	// Открываем файл для записи
	SimpleWaveWriter wave_writer(true);
	if(write_wave_file)
	{
		if(!file_name.empty())
			wave_writer.FileOpen(file_name.c_str());
		else
			std::cout << "ERROR: File name is empty" <<std::endl;
	}

	// Создание устройства
	std::cout << "Start test..." <<std::endl;
	FoxxWire::SDK::ITr705 *tr705 = FoxxWire::SDK::CreateTr705();
	if(!tr705)
		return EXIT_FAILURE;

	// Инициализация устройства
	FoxxWire::SDK::DeviceInitSettings init_settings;
	init_settings.PeerAddr = device_list[0].UnitAddr;
	init_settings.PeerJsonPort = device_list[0].UnitPort;
	init_settings.HostAddr = host_addr;
	init_settings.HostJsonPort = host_port;

	if (!tr705->Init(init_settings))
		return EXIT_FAILURE;

	// Информация об устройстве
	FoxxWire::SDK::DeviceInfo tr705_info = tr705->GetDeviceInfo();
	std::cout << "Id: " << tr705_info.UnitId.c_str() << "  Channels: " << tr705_info.NumInputs << "/0 "
				" Sample freq: " << tr705_info.SampleFreq << std::endl;
	std::cout << "Number cycle samples: " << tr705->GetNumCycleSamples() << std::endl;

	if(channel_testing >= tr705_info.NumInputs * 2)
		channel_testing = 0;

	// Установка FM частоты
	std::vector<uint32_t> fm_freqs = {100500, 103700, 105300, 90100};
	bool res = tr705->SetUnitRadioFreqs(fm_freqs.data());
	std::cout << "SetUnitRadioFreqs test result = " << res << std::endl;

	// Получение RDS
	std::vector< FoxxWire::SDK::Tr705TunerRdsDataValues > rds_vals;
	rds_vals.resize( tr705_info.NumInputs );
	res = tr705->GetUnitRdsData(rds_vals.data(), true);
	std::cout << "GetUnitRdsData test result = " << res << std::endl;

	// Запуск RTP потоков
	FoxxWire::SDK::DeviceStartSettings start_settings;
	start_settings.HostRtpPort = 10002;
	start_settings.RtpExchangeCallback = [&wave_writer, channel_testing]( const std::vector< float * >& recv_buffers, size_t num_samples ){
		if(wave_writer.IsOpened())
			// Запись одного канала
			wave_writer.WriteData(recv_buffers[channel_testing], num_samples * sizeof( float ));
	};
	tr705->Start(start_settings);

	// Главный цикл
	std::vector< uint32_t > freqs;
	std::vector< FoxxWire::SDK::Tr705TunerQualityValues > rqvals;
	std::vector< float > levels;

	freqs.resize( tr705_info.NumInputs );
	rqvals.resize( tr705_info.NumInputs );
	fm_freqs[ 0 ] = fm_freqs[ 1 ] = fm_freqs[ 2 ] = fm_freqs[ 3 ] = 0;

	long written_cycle = 0;
	while (!app_stop_flag)
	{
		std::this_thread::sleep_for(std::chrono::milliseconds(10));

		// Обновляем информацию о частоте и качестве на экране (один раз на 8 цикла)
		if (!(written_cycle & 0x07))
		{
			for( auto& f : freqs ) f = ~0u;
			for( auto& q : rqvals ) q.Clear( );

			bool ok_freqs = false;
			bool ok_quals = false;
			if( tr705->GetDeviceConnectState() == FoxxWire::SDK::DeviceConnectState::RtpActive )
			{
				ok_freqs = tr705->GetUnitRadioFreqs( freqs.data( ), true );
				ok_quals = tr705->GetUnitRadioQuality( rqvals.data( ), true );
			}

			std::cout << " (" << (ok_freqs?"+":"-") << ")F=" << freqs[ 0 ];
			std::cout << " (" << (ok_quals?"+":"-") << ")Q: RSSI=" << rqvals[ 0 ]._RSSI << " SNR=" << rqvals[ 0 ]._SNR << " Multipath=" << rqvals[ 0 ]._MultiPath << " Pilot=" << rqvals[ 0 ]._Pilot << " Valid=" << rqvals[ 0 ]._Valid << std::endl;
		}

		++written_cycle;
	}

	// Заканчиваем запись в файл
	if( wave_writer.IsOpened() )
		wave_writer.FileClose(( unsigned )tr705_info.SampleFreq );

	return EXIT_SUCCESS;
}


