#pragma once

// import C/C++ stdlib
#include <cstdint>
#include <string>
#include <functional>
#include <vector>

// API symbols
#include "_sdk705-api.h"

// warning control
#if defined( _MSC_VER )
#pragma warning( push ) 
#pragma warning( disable : 4251 ) /* : class 'ENCLOSED' needs to have dll-interface to be used by clients of class 'CONTAINING' */
#endif /*defined( _MSC_VER )*/


namespace FoxxWire::SDK
{

	// --------------------------------------------------------------------------------------
	//  Состояния устройства FoxxWire в объекте устройства
	// --------------------------------------------------------------------------------------

	enum class DeviceConnectState : uint32_t
	{
		Disconnected = 0,    // объект устройства сконструирован
		Connected,           // объект устройства выполнил успешный Init() и обменялся сообщениями с устройством
		RtpActive,           // объект устройства выполнил успешный Start() и запустил RTP обмен с устройством
	};

	// --------------------------------------------------------------------------------------
	//  предобъявления типов для основного интерфейса и фабрики
	// --------------------------------------------------------------------------------------

	struct DeviceInitSettings;       // параметры для IDevice::Init()
	struct DeviceStartSettings;      // параметры для IDevice::Start()
	struct DeviceInfo;               // информация об устройстве
	struct DeviceInfoEx;             // информация об устройстве c адресом устройства
	struct Tr705TunerQualityValues;  // аргумент для TR-705 интерфейсов
	struct Tr705TunerRdsDataValues;  // аргумент для TR-705 интерфейсов

	// --------------------------------------------------------------------------------------
	//
	//  Функция для поиска устройств FoxxWire в сети
	//
	// --------------------------------------------------------------------------------------

	typedef  std::vector< DeviceInfoEx >  DevicesList;

	_API_SDK705_  extern   bool  FindActiveDevices( const std::string& host_ip_addr, uint32_t host_port,
													const std::string& broadcast_ip_addr, uint32_t broadcast_port,
													DevicesList& );

	// --------------------------------------------------------------------------------------
	//
	//  Интерфейсный класс для работы с устройством ТР-705
	//
	// --------------------------------------------------------------------------------------

	class  _API_SDK705_  ITr705
	{

	public:
		virtual ~ITr705() = default;

		// Открытие устройства, установка связи, получение информации об устройстве
		virtual bool Init(const DeviceInitSettings &settings) = 0;

		// Закрытие устройства
		virtual void Deinit() = 0;

		// Запуск RTP-обмена
		virtual bool Start(const DeviceStartSettings &settings) = 0;

		// Остановка RTP-обмена
		virtual void Stop() = 0;

	public:  // Можно вызывать всегда
		// Получить состояние устройства
		virtual DeviceConnectState GetDeviceConnectState() const = 0;

	public:  // Можно вызывать после успешного Init()

		// Получить уникальный идентификатор устройства в UTF-8
		virtual std::string GetDeviceId() const = 0;

		// Получить ссылку на структуру с информацией об устройстве
		virtual DeviceInfo GetDeviceInfo() const = 0;

		// Получить количество тюнеров (они же каналы записи/стереопары) устройства
		virtual uint32_t GetNumRecordChannels() const = 0;

		// Получить размер буферов в сэмплах для одного цикла обмена
		// (для одного канала стереопары; таковы размеры буферов у callback для RTP-обмена)
		virtual uint32_t GetNumCycleSamples() const = 0;

		// Получить частоту дискретизации, на которой работает устройство
		virtual uint32_t GetSampleFreq() const = 0;

	public:
		// запрос частоты приема в килогерцах для всех тюнеров платы
		// (буфер 'freqs' должен иметь размер равный 'DeviceInfo::NumInputs', либо GetNumRecordChannels())
		virtual  bool  GetUnitRadioFreqs( uint32_t *freqs, bool update_from_device ) = 0;

		// установка частоты приема в килогерцах для всех тюнеров платы
		// если freqs[j]==0 то для тюнера j настройка не производится
		// (буфер 'freqs' должен иметь размер равный 'DeviceInfo::NumInputs', либо GetNumRecordChannels())
		virtual  bool  SetUnitRadioFreqs( const uint32_t *freqs ) = 0;

		// запрос параметров качества приема для всех тюнеров (размер буфера 'DeviceInfo::NumInputs', либо GetNumRecordChannels())
		virtual  bool  GetUnitRadioQuality( Tr705TunerQualityValues *unit_values, bool update_from_device ) = 0;

		// запрос данных RDS для всех тюнеров (размер буфера 'DeviceInfo::NumInputs', либо GetNumRecordChannels())
		virtual  bool  GetUnitRdsData( Tr705TunerRdsDataValues *unit_values, bool update_from_device ) = 0;
	};

	// --------------------------------------------------------------------------------------
	//
	//  Основная фабрика для объектов устройства ТР-705
	//
	//  После создания объекта нужного типа для начала работы с устройством его необходимо
	//  открыть вызовом метода Init() с передачей IP-адресов
	//
	//  Если необходимо запустить обмен аудиопотоками по RTP, то после успешного вызова Init()
	//  необходимо далее вызвать метод Start() с передачей ему портов для сокетов обмена и
	//  метода обратного вызова для обмена данными аудиопотоков с приложением
	//
	// --------------------------------------------------------------------------------------

	_API_SDK705_   extern   ITr705  *CreateTr705( );

	// --------------------------------------------------------------------------------------
	//
	//  Структуры параметров и прочие вспомогательные типы для классов FoxxWire
	//
	// --------------------------------------------------------------------------------------

	// --------------------------------------------------------------------------------------
	//  Параметры для метода инициализации объекта устройства (для метода IDevice::Init())
	// --------------------------------------------------------------------------------------

	struct  _API_SDK705_  DeviceInitSettings
	{
		// IP-адрес устройства в UTF-8
		std::string PeerAddr;

		// порт устройства для обмена JSON-сообщениями
		uint16_t PeerJsonPort = 15001;

		// IP-адрес нашего интерфейса для обмена с устройством в UTF-8
		std::string HostAddr;

		// наш порт для обмена JSON-сообщениями
		uint16_t HostJsonPort = 0;
	};


	// --------------------------------------------------------------------------------------
	//  Параметры для запуска RTP-обмена объекта устройства (для метода IDevice::Start())
	// --------------------------------------------------------------------------------------

	struct  _API_SDK705_  DeviceStartSettings
	{
		// вспомогательный тип == вектор указателей на набор планарных канальных буферов
		typedef  std::vector< float * >  buffer_ptrs;

		// наш базовый порт для приема и отправки RTP-пакетов
		// если потоков больше одного, то последующие порты идут с шагом 2
		uint16_t  HostRtpPort;

		// максимальная разрешенная задержка входящих RTP-пакетов в штуках
		uint32_t  RecvDelayMax = 20;

		// типичная задержка входящих RTP-пакетов в штуках (размер входной очереди пакетов)
		uint32_t  RecvDelayNorm	= 10;  // 10 пакетов == 40 миллисекунд

		// callback приложения для обмена блоками аудиопотоков
		// обмен аудио блоками происходит в планарном формате float 32 бита
		// (scattered, т.е. с отдельными указателями на каждый буфер канала)
		// за один вызов callback должен принять и отдать по 'num_samples' сэмплов в каждом канале
		// вектор 'recv_buffers' содержит указатели на буферы моноканалов
		// размер вектора 'recv_buffers' равен удвоенной величине GetNumRecordChannels()
		// 'num_samples' равно величине GetNumCycleSamples(), возвращаемой объектом устройства
		// если поле пустое, то ни один из callback'ов приложения не вызывается, но RTP обмен
		// будет запущен: входящий RTP поток для записи теряется, исходящий RTP поток заполняется нулями
		typedef  void  RtpExchangeCallbackFunc( const buffer_ptrs& recv_buffers, size_t num_samples );
		std::function< RtpExchangeCallbackFunc >  RtpExchangeCallback;

		// callback приложения для получения указателей на буферы записи приложения 'recv_buffers'
		// если оба callback'а ('RtpExchangeCallback' и данное поле) непустые, то управление буферами записи
		// осуществляется приложением. Для получения указателей на буферы 'recv_buffers' данный callback
		// будет вызван первым. Далее, после заполнения буферов записи из RTP будет вызван callback
		// обмена данными RtpExchangeCallback.
		typedef  void  RtpGetRecBufCallbackFunc( buffer_ptrs& recv_buffers, size_t num_samples );
		std::function< RtpGetRecBufCallbackFunc >  RtpGetRecBufCallback;
	};


	// --------------------------------------------------------------------------------------
	//  Структура с информацией об устройстве FoxxWire, которую сообщает само устройство
	//  Копия хранится в объекте устройства и заполняется информацией при выполнении Init()
	//  в процессе обмена JSON сообщениями с устройством
	//  Эта структура доступна с помощью вызова IDevice::GetDeviceInfo() после успешного
	//  выполнения Init(), отдельные части доступны с помощью других методов IDevice
	// --------------------------------------------------------------------------------------

	struct  _API_SDK705_  DeviceInfo
	{
		// уникальный идентификатор устройства в UTF-8
		std::string  UnitId;
		// версия платы
		uint32_t      HardwareVersion = 0;
		// версия прошивки платы
		uint32_t      FirmwareVersion = 0;
		// версия программного обеспечения платы/устройства (если применимо, например, TR-7)
		uint32_t      SoftwareVersion = 0;
		// версия RTP-протокола (note: в версии 1 были испорченные PayloadType в заголовках, см RtpPacket.h)
		uint32_t      AudioProtocolVersion = 0;
		// версия JSON-протокола (описание в Wiki проекта ASIO драйвера)
		uint32_t      ControlProtocolVersion = 0;
		// номер первого RTP порта устройства для проигрывания; если потоков больше одного, то последующие порты идут с шагом 2
		uint32_t      PeerRtpPort = 0;
		// количество стереоканалов записи с платы/устройства (оно же обычно равно кол-ву RTP-потоков от устройства)
		uint32_t      NumInputs = 0;
		// частота дискретизации, на которой работает устройство (для большинства будет всегда 48000, в TR322 м.б. 44100)
		uint32_t      SampleFreq = 0;
	};


	// --------------------------------------------------------------------------------------
	//  Структура с информацией об устройстве FoxxWire, аргумент для функции поиска устройств
	// --------------------------------------------------------------------------------------

	struct  _API_SDK705_  DeviceInfoEx  :  DeviceInfo
	{
		// IP-адрес устройства в UTF-8
		std::string UnitAddr;

		// порт устройства для обмена JSON-сообщениями
		uint16_t UnitPort = 15001;
	};


	// --------------------------------------------------------------------------------------
	//  Структура для параметров метода ITr705
	// --------------------------------------------------------------------------------------

	struct  _API_SDK705_  Tr705TunerQualityValues
	{
		uint32_t  _RSSI      = 0;  // [ 0...255 ] : Received Signal Strength Indicator
		uint32_t  _SNR       = 0;  // [ 0...255 ] : Signal to Noise Ratio
		uint32_t  _MultiPath = 0;  // [ 0...255 ] : Что это такое и в чем измеряется неизвестно
		uint32_t  _Pilot     = 0;  // [ 0 / 1   ] : Stereo Pilot Indicator
		uint32_t  _Valid     = 0;  // [ 0 / 1   ] : Valid Indicator
		// ---
		void  Clear( ){ _RSSI = _SNR = _MultiPath = _Pilot = _Valid = 0; }
	};

	// --------------------------------------------------------------------------------------

	struct  _API_SDK705_  Tr705TunerRdsDataValues
	{
		std::wstring  _RdsPSrv;  // "Program Service Name"
		std::wstring  _RdsText;  // "RDS Text"
		std::wstring  _RdsTime;  // "RDS Time"
		// ---
		void  Clear( ){ _RdsText.clear( ); _RdsPSrv.clear( ); _RdsTime.clear( ); }
	};

	// --------------------------------------------------------------------------------------

} // namespace FoxxWire

#if defined( _MSC_VER )
#pragma warning( pop )
#endif /*defined( _MSC_VER )*/


